<?php

/**
 * @file controllers/modals/publish/PublishHandler.php
 *
 * Copyright (c) 2014-2021 Simon Fraser University
 * Copyright (c) 2003-2021 John Willinsky
 * Distributed under the GNU GPL v3. For full terms see the file docs/COPYING.
 *
 * @class PublishHandler
 *
 * @ingroup controllers_modals_publish
 *
 * @brief A handler to load final publishing confirmation checks
 */

namespace PKP\controllers\modals\publish;

use APP\components\forms\publication\PublishForm;
use APP\core\Application;
use APP\facades\Repo;
use APP\handler\Handler;
use APP\publication\Publication;
use APP\submission\Submission;
use APP\template\TemplateManager;
use PKP\core\JSONMessage;
use PKP\core\PKPApplication;
use PKP\core\PKPRequest;
use PKP\security\authorization\PublicationAccessPolicy;
use PKP\security\authorization\SubmissionAccessPolicy;
use PKP\security\Role;

class PublishHandler extends Handler
{
    /** @var Submission */
    public $submission;

    /** @var Publication */
    public $publication;

    /**
     * Constructor.
     */
    public function __construct()
    {
        parent::__construct();
        $this->addRoleAssignment(
            [Role::ROLE_ID_SUB_EDITOR, Role::ROLE_ID_MANAGER, Role::ROLE_ID_SITE_ADMIN, Role::ROLE_ID_ASSISTANT],
            ['publish']
        );
    }


    //
    // Overridden methods from Handler
    //
    /**
     * @copydoc PKPHandler::initialize()
     */
    public function initialize($request, $args = null)
    {
        parent::initialize($request, $args);
        $this->submission = $this->getAuthorizedContextObject(Application::ASSOC_TYPE_SUBMISSION);
        $this->publication = $this->getAuthorizedContextObject(Application::ASSOC_TYPE_PUBLICATION);
        $this->setupTemplate($request);
    }

    /**
     * @copydoc PKPHandler::authorize()
     */
    public function authorize($request, &$args, $roleAssignments)
    {
        $this->addPolicy(new SubmissionAccessPolicy($request, $args, $roleAssignments));
        $this->addPolicy(new PublicationAccessPolicy($request, $args, $roleAssignments));
        return parent::authorize($request, $args, $roleAssignments);
    }


    //
    // Public handler methods
    //
    /**
     * Display a publishing confirmation form
     *
     * @param array $args
     * @param PKPRequest $request
     *
     * @return JSONMessage JSON object
     */
    public function publish($args, $request)
    {
        $templateMgr = TemplateManager::getManager($request);

        $submissionContext = $request->getContext();
        if (!$submissionContext || $submissionContext->getId() !== $this->submission->getData('contextId')) {
            $submissionContext = app()->get('context')->get($this->submission->getData('contextId'));
        }

        $primaryLocale = $submissionContext->getPrimaryLocale();
        $allowedLocales = $submissionContext->getSupportedSubmissionLocales();
        $errors = Repo::publication()->validatePublish($this->publication, $this->submission, $allowedLocales, $primaryLocale);

        $publicationApiUrl = $request->getDispatcher()->url($request, PKPApplication::ROUTE_API, $submissionContext->getPath(), 'submissions/' . $this->submission->getId() . '/publications/' . $this->publication->getId() . '/publish');

        $publishForm = new PublishForm($publicationApiUrl, $this->publication, $submissionContext, $errors);

        $settingsData = [
            'components' => [
                PublishForm::FORM_PUBLISH => $publishForm->getConfig(),
            ],
        ];

        $templateMgr->registerClass(PublishForm::class, PublishForm::class);
        $templateMgr->assign('publishData', $settingsData);

        return $templateMgr->fetchJson('controllers/modals/publish/publish.tpl');
    }
}
